<?php

declare(strict_types=1);

use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\PulsePs\Entity\PulseContact;
use PrestaShop\Module\PulsePs\Install\InstallerFactory;
use PrestaShop\Module\PulsePs\Service\Pulse;
use GuzzleHttp\Client;

if (!defined('_PS_VERSION_')) {
    exit;
}

// need it because InstallerFactory is not autoloaded during the install
require_once __DIR__.'/vendor/autoload.php';

class PulsePs extends Module
{
    private $pulse;

    public function __construct(Pulse $pulse)
    {
        $this->name = 'pulseps';
        $this->author = 'Utixo';
        $this->version = '1.0.0';
        $this->ps_versions_compliancy = ['min' => '1.7.7.0', 'max' => _PS_VERSION_];
        $this->bootstrap = true;

        $this->pulse = $pulse;

        if(!$this->pulse->pulseAuthExists()){
            $this->warning = strip_tags(Pulse::NOT_CONNECTED_WARNING);
        }

        parent::__construct();

        $this->displayName = $this->l('Utixo Pulse');
        $this->description = $this->l('Sync contacts to your Pulse account');
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $installer = InstallerFactory::create();

        return $installer->install($this);
    }

    public function uninstall()
    {
        $installer = InstallerFactory::create();

        return $installer->uninstall() && parent::uninstall();
    }

    public function getContent()
    {
        $output = null;

        if (Tools::isSubmit('submit'.$this->name)) {
            $hostname = strval(Tools::getValue(Pulse::HOSTNAME));
            $client_id = strval(Tools::getValue(Pulse::CLIENT_ID));
            $client_secret = strval(Tools::getValue(Pulse::CLIENT_SECRET));
            $group_id = strval(Tools::getValue(Pulse::GROUP_ID));

            if (
                !$hostname ||
                !$client_id ||
                !$client_secret
            ) {
                $output .= $this->displayError($this->l('Invalid Configuration value'));
            } else {
                Configuration::updateValue(Pulse::HOSTNAME, $hostname);
                Configuration::updateValue(Pulse::CLIENT_ID, $client_id);
                Configuration::updateValue(Pulse::CLIENT_SECRET, $client_secret);
                Configuration::updateValue(Pulse::GROUP_ID, $group_id);
                $output .= $this->displayConfirmation($this->l('Settings updated'));
            }
        }

        $output .= $this->displayForm();

        if(!$this->pulse->pulseAuthExists()){
            $this->context->controller->warnings[] = Pulse::NOT_CONNECTED_WARNING;
        }

        return $output;
    }

    public function displayForm()
    {
        $fieldsValue = [
            Pulse::HOSTNAME => Tools::getValue(Pulse::HOSTNAME, $this->pulse->getHostname()),
            Pulse::CLIENT_ID => Tools::getValue(Pulse::CLIENT_ID, $this->pulse->getClientId()),
            Pulse::CLIENT_SECRET => Tools::getValue(Pulse::CLIENT_SECRET, $this->pulse->getClientSecret()),
            '_' . Pulse::HOSTNAME => Tools::getValue(Pulse::HOSTNAME, $this->pulse->getHostname()),
            '_' . Pulse::CLIENT_ID => Tools::getValue(Pulse::CLIENT_ID, $this->pulse->getClientId()),
            '_' . Pulse::REDIRECT_URI => _PS_BASE_URL_ . (new Link)->getAdminLink('PulseOauthCallback'),
            '_' . Pulse::SCOPES => Pulse::SCOPES_VALUE,
        ];
        if($this->pulse->pulseAuthExists()){
            $fieldsValue[Pulse::GROUP_ID] = Tools::getValue(Pulse::GROUP_ID, $this->pulse->getGroupId());
        }
        $form = [
            'form' => [
                'legend' => [
                    'title' => $this->trans('Parameters', [], 'Modules.Pulse.Admin'),
                    'icon' => 'icon-cogs'
                ],
                'input' => [
                    [
                        'type' => 'text',
                        'label' => $this->trans(
                            'Pulse hostname',
                            [],
                            'Modules.Pulse.Admin'
                        ),
                        'desc' => $this->trans(
                            "ex. https://my.pulse-marketing.net",
                            [],
                            'Modules.Pulse.Admin'
                        ),
                        'name' => Pulse::HOSTNAME,
                        'required' => true,
                        'size' => '128'
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->trans(
                            'Pulse Client ID',
                            [],
                            'Modules.Pulse.Admin'
                        ),
                        'desc' => $this->trans(
                            "Create an API client id on Pulse and paste it here",
                            [],
                            'Modules.Pulse.Admin'
                        ),
                        'name' => Pulse::CLIENT_ID,
                        'required' => true,
                        'size' => '128'
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->trans(
                            'Pulse Secret',
                            [],
                            'Modules.Pulse.Admin'
                        ),
                        'desc' => $this->trans(
                            "Paste the secret key for your API client id",
                            [],
                            'Modules.Pulse.Admin'
                        ),
                        'name' => Pulse::CLIENT_SECRET,
                        'required' => true,
                        'size' => '128'
                    ],
                    [
                        'type' => 'hidden',
                        'name' => '_' . Pulse::HOSTNAME,
                    ],
                    [
                        'type' => 'hidden',
                        'name' => '_' . Pulse::CLIENT_ID,
                    ],
                    [
                        'type' => 'hidden',
                        'name' => '_' . Pulse::REDIRECT_URI,
                    ],
                    [
                        'type' => 'hidden',
                        'name' => '_' . Pulse::SCOPES,
                    ]
                ],
                'submit' => [
                    'title' => $this->trans('Save', [], 'Admin.Actions'),
                    'class' => 'btn btn-default pull-right'
                ],
            ],
        ];

        $callback_url = $this->context->link->getModuleLink($this->name, 'oauth');

        if($this->pulse->hostnameExists() && $this->pulse->clientIdExists() && $this->pulse->clientSecretExists() && !$this->pulse->pulseAuthExists()){
            $form['form']['buttons'][] =
                [
                    'href' => $fieldsValue[Pulse::HOSTNAME].'/authorize',
                    'title' => 'Connect to Pulse',
                    'icon' => 'process-icon-refresh',
                    'id' => 'connect_pulse',
                ];
        } elseif ($this->pulse->pulseAuthExists()){
            $form['form']['buttons'][] =
                [
                    'href' => (new Link)->getAdminLink('PulseOauthDisconnect'),
                    'title' => 'Disconnect Pulse',
                    'icon' => 'process-icon-cancel',
                    'id' => 'disconnect_pulse',
                ];

            try{
                $result = $this->pulse->getContactGroups();
                $contactGroups = $result['data']['contact_groups'];

                $form['form']['input'][] = [
                    'label' => $this->trans('Contact group', [], 'Modules.Pulseps.Admin'),
                    //'hint' => $this->trans('Choose a group to sync contacts to', [], 'Modules.Pulseps.Admin'),
                    'name' => Pulse::GROUP_ID,
                    'type' => 'select',
                    'options' => [
                        'query' => $contactGroups,
                        'id' => 'id',
                        'name' => 'name',
                    ]
                ];
            } catch (\PrestaShop\Module\PulsePs\Service\Exception\PulseException $e){
                //dump($e->getMessage());die;
            }


        }

        $helper = new HelperForm();
        $helper->table = $this->table;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->submit_action = 'submit'.$this->name;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = [
            'fields_value' => $fieldsValue,
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id
        ];

        return $helper->generateForm([$form]);
    }

    /**
     * Add buttons to main buttons bar
     */
    public function hookActionCustomerAccountAdd(array $params)
    {
        if($this->pulse->pulseAuthExists()) {
            $customer = $params['newCustomer'];
            PrestaShopLogger::addLog('Pulse - Attempting sync of new Customer ID: '.$customer->id, 1);
            $this->syncContact($customer);
        }
    }


    public function hookActionCustomerAccountUpdate(array $params)
    {
        if($this->pulse->pulseAuthExists()){
            $customer = $params['customer'];
            PrestaShopLogger::addLog('Pulse - Attempting sync of Customer ID: '.$customer->id, 1);
            $this->syncContact($customer);
        }
    }

    private function syncContact($customer)
    {
        $receive_emails = ($customer->newsletter == 1) ? true : false;
        $contactGroups = [
            $this->pulse->getGroupId()
        ];

        /** @var EntityManagerInterface $entityManager */
        $entityManager = $this->getContainer()->get('doctrine.orm.entity_manager');

        $pulse_contact = null;
        if($pulse_contact = $this->pulse->getContactByLocalId((int) $customer->id)){
            try {
                // If email changed from the one stored on Pulse
                if ($pulse_contact['email'] != $customer->email){
                    // If new email already exists in pulse, delete that old contact first
                    if ($old_pulse_contact = $this->pulse->getContactByEmail($customer->email)) {
                        $this->pulse->deleteContact($old_pulse_contact['id']);
                    }
                }
                $this->pulse->updateContact($pulse_contact['id'], $customer->email, $customer->firstname, $customer->lastname, null, (int) $customer->id, $contactGroups, $receive_emails);
                PrestaShopLogger::addLog('Pulse - Updated Pulse Contact ID: '.$pulse_contact['id'].', identified by Customer ID: '.$customer->id, 1);
                $pulse_contact = $this->pulse->getContact($pulse_contact['id']);
            } catch (\PrestaShop\Module\PulsePs\Service\Exception\PulseException $e){
                PrestaShopLogger::addLog($e->getMessage(), 3);
            }
        } else {
            try{
                // If email already exists in pulse, update old contact
                if ($pulse_contact = $this->pulse->getContactByEmail($customer->email)) {
                    $this->pulse->updateContact($pulse_contact['id'], $customer->email, $customer->firstname, $customer->lastname, null, (int) $customer->id, $contactGroups, $receive_emails);
                    PrestaShopLogger::addLog('Pulse - Updated Pulse Contact ID: '.$pulse_contact['id'].' with Customer ID: '.$customer->id, 1);
                    $pulse_contact = $this->pulse->getContact($pulse_contact['id']);
                } else {
                    $pulse_contact_id = $this->pulse->createContact($customer->email, $customer->firstname, $customer->lastname, null, (int) $customer->id, $contactGroups, $receive_emails);
                    PrestaShopLogger::addLog('Pulse - Created Pulse Contact ID: '.$pulse_contact_id.' for Customer ID: '.$customer->id, 1);
                    $pulse_contact = $this->pulse->getContact($pulse_contact_id);
                }

            } catch (\PrestaShop\Module\PulsePs\Service\Exception\PulseException $e){
                PrestaShopLogger::addLog($e->getMessage(), 3);
            }
        }
    }

    /**
     * Add buttons to main buttons bar
     */
    public function hookDisplayAdminForm(array $params)
    {
        if($params['smarty']->tpl_vars['current']->value === "index.php?controller=AdminModules&configure=".$this->name){
            $this->context->controller->addJS($this->_path . 'views/dist/pulse_auth.js');
        }
    }

    /**
     * Add global messages
     */
    public function hookDisplayBackOfficeTop(array $params)
    {
        switch ($this->context->controller->controller_name){
            case 'AdminDashboard':
            case 'AdminModulesManage':
                if(!$this->pulse->pulseAuthExists()){
                    $this->context->controller->warnings[] = Pulse::NOT_CONNECTED_WARNING;
                }
                break;
        }
    }

    /**
     * Render a twig template.
     */
    private function render(string $template, array $params = []): string
    {
        /** @var Twig_Environment $twig */
        $twig = $this->get('twig');

        return $twig->render($template, $params);
    }

    /**
     * Get path to this module's template directory
     */
    private function getModuleTemplatePath(): string
    {
        return sprintf('@Modules/%s/views/templates/admin/', $this->name);
    }
}
