<?php
/**
* NOTICE OF LICENSE
*
* This file is licenced under the Software License Agreement.
* With the purchase or the installation of the software in your application
* you accept the licence agreement.
*
* You must not modify, adapt or create derivative works of this source code
*
*  @author    Poste Italiane
*  @copyright 2010-2020 Poste Italiane SPA
*  @license   LICENSE.txt
*/

if (!defined('_PS_VERSION_'))
	exit;

class PosteItaliane extends CarrierModule
{
	private $_html = '';
	private $_postErrors = array();
	private $_moduleName = 'PosteItaliane';

	const CONFIGURED                       = 0;
	const ERR_STORE_ADDRESS_NOT_CONFIGURED = 1;
	const ERR_MODULE_CONFLICT              = 2;

	public function __construct()
	{
		$this->name                   = 'posteitaliane';
		$this->tab                    = 'shipping_logistics';
		$this->version                = '1.0.7';
		$this->author                 = 'PosteItaliane';
		$this->ps_versions_compliancy = array('min' => '1.5', 'max' => '1.7');
		$this->module_key             = '72d9a5eab342be2373c78b4a1b5abbf5';

		$this->displayName = 'Poste Italiane';
		$this->description = 'Poste Italiane e-Commerce Shipping Solutions è il modulo ufficiale di Poste per utilizzare i servizi di spedizione in Italia e all’estero sulla tua piattaforma in back office e front end. Zero costi di sviluppo e attivazione!';
		$this->bootstrap = true;

		$this->context = Context::getContext();
		
		parent::__construct();
	}

	public function isAdmin(){
		return (new Cookie('psAdmin'))->id_employee;
	}

	public function install()
	{
		if (!parent::install())
			return false;

		$this->registerHook('actionValidateOrder');
		$this->registerHook('displayBeforeCarrier');

		$version = explode('.', _PS_VERSION_);
		
		if ( $version[0] === '1' && $version[1] <= 7 && $version[2] < 7 ) { // under 1.7.7
			$hookAdminOrder = 'Left';
		} else {
			$hookAdminOrder = 'Main';
		}

		$this->registerHook("displayAdminOrder{$hookAdminOrder}");

		return true;
	}
	
	public function uninstall()
	{
		// Uninstall
		if (!parent::uninstall())
			return false;

		$this->uninstallCarriers();
		
		return true;
	}
	
	public function getContent()
	{
		$error = FALSE;
		$output = "";
		
		$checkField = function($name) use (&$output, &$error) {
			$fieldValue = Tools::getValue($name);

			if (
				stripos($name, 'metodidispedizione') === FALSE &&
				stripos($name, 'attivaservizio') === FALSE &&
				empty($fieldValue)
			) {
				$output .= $this->displayError($this->l('Riempi tutti i campi!'));
				$error = TRUE;
			}

			else Configuration::updateValue($name, $fieldValue);
		};

		$addConfirmationMessage = function($message) use (&$output) {
			$output .= $this->displayConfirmation($this->l($message));
		};

		$addErrorMessage = function($message) use (&$output) {
			$output .= $this->displayError($this->l($message));
		};
	 
		if (Tools::isSubmit('submit' . $this->name . 'CheckoutCarriers'))
		{
			foreach ($this->getCheckoutCarriers() as $checkoutCarrier) {	
				$configurationKeyName = $checkoutCarrier["configuration_key_name"];
				$metodoSpedizioneCurrVal = Configuration::get($configurationKeyName);
				$metodoSpedizioneNewVal = Tools::getValue($configurationKeyName);

				if ($metodoSpedizioneCurrVal != $metodoSpedizioneNewVal) {
					if ($metodoSpedizioneNewVal) $this->getAndInstallCarriers([$checkoutCarrier["name"]]);
					else $this->uninstallCarrier($checkoutCarrier["name"]);
				}

				$checkField($configurationKeyName);
			}
		}
	 
		if (Tools::isSubmit('submit' . $this->name . 'Settings'))
		{			
			$checkField('posteitaliane_codicepostazione');
			$checkField('posteitaliane_codicecliente');
			$checkField('posteitaliane_login');
			$checkField('posteitaliane_password');

			if (!$error) {
				$serviceId = Tools::getValue("posteitaliane_servizio") - 1;
				$serviceName = $this->GetActivableServices()[$serviceId];
				$carrierName = "POSTEITALIANE";

				if ($serviceName == "SDA - Extralarge") {
					$serviceId = 0;
					$carrierName = "SDA";
				}
	
				$request = array(
					"Method" => "AddCarrier",
					"Params" => array(
						"CarrierName" => $carrierName,
						"Credentials" => array(
							"postazione"     => Tools::getValue("posteitaliane_codicepostazione"),
							"codice_cliente" => Tools::getValue("posteitaliane_codicecliente"),
							"login"          => Tools::getValue("posteitaliane_login"),
							"password"       => Tools::getValue("posteitaliane_password"),
							"service"        => $serviceId,
							"label_type"     => 0,
							"return_label"   => 0,
							"label"          => $serviceName
						)
					)
				);
		
				$addCarrierApiResponse = $this->ApiRequest($request);
	
				if ($addCarrierApiResponse->Result == "OK") $addConfirmationMessage('Configurazione aggiunta con successo!');
				else $addErrorMessage('Errore nell\'aggiunta della configurazione: ' . $addCarrierApiResponse->ErrorMessage);
			}
		}
	 
		if (Tools::isSubmit('submit' . $this->name . 'RequestAccount'))
		{		
			$checkField('posteitaliane_nomecognome');
			$checkField('posteitaliane_ragionesociale');
			$checkField('posteitaliane_indirizzo');
			$checkField('posteitaliane_citta');
			$checkField('posteitaliane_cap');
			$checkField('posteitaliane_nazione');
			$checkField('posteitaliane_partitaiva');
			$checkField('posteitaliane_telefono');
			$checkField('posteitaliane_email');
			
			$activateServices = [];

			foreach ($this->GetActivableServices() as $activateServiceIndex => $activateService) {
				$postServiceId = $activateServiceIndex + 1;
				
				$postKeyName = "posteitaliane_attivaservizio_" . $postServiceId;

				if (!empty(Tools::getValue($postKeyName))) {
					$activateServices[] = $activateService;
					unset($_POST[$postKeyName]);
				}
			}

			if (empty($activateServices)) {
				$addErrorMessage('Seleziona almeno un servizio!');
				$error = TRUE;
			}

			$shopPostcode = Tools::getValue("posteitaliane_cap");

			if (!is_numeric($shopPostcode) || Tools::strlen($shopPostcode) != 5) {
				$addErrorMessage('Il cap deve essere di cinque caratteri');
				$error = TRUE;
			}

			$shopVATNumber = Tools::getValue("posteitaliane_partitaiva");

			if (!is_numeric($shopVATNumber) || Tools::strlen($shopVATNumber) != 11) {
				$addErrorMessage('La partita IVA deve essere numerica e di 11 caratteri');
				$error = TRUE;
			}

			if (!$error) {
				$shopName               = Configuration::get('PS_SHOP_NAME');
				$shopEmail              = Configuration::get('PS_SHOP_EMAIL');
				$shopCompany            = Tools::getValue("posteitaliane_ragionesociale");
				$activateServicesString = implode(", ", $activateServices);
				$shopFieldsStringChunks = [];

				foreach ($_POST as $postKey => $postValue)
				{
					if (
						stripos($postKey, 'submit') !== FALSE ||
						stripos($postKey, 'posteitaliane_attivaservizio_') !== FALSE ||
						$postKey == "tab"
					) continue;

					$postKeyName = str_replace("posteitaliane_", "", $postKey);

					$shopFieldsStringChunks[] = $postKeyName . ": " . $postValue;
				}

				$shopFieldsString = implode("<br>", $shopFieldsStringChunks);

				$emailBody = <<<emailBody
Buongiorno,<br><br>
il merchant {$shopName} (email: {$shopEmail}) ha richiesto un contatto commerciale tramite il plugin spedizioni di Poste Italiane su PrestaShop	<br><br>
Il merchant è interessato al prodotto {$activateServicesString}<br><br>
A seguire le informazioni di dettaglio del merchant:<br><br>
{$shopFieldsString}
emailBody;
				
				$request = array(
					"Method" => "PosteItalianeSendEmailMessage",
					"Params" => array(
						"Subject" => "Plug-in spedizioni – Lead - " . $shopEmail . ' - ' . $shopCompany . ' - ' . $shopVATNumber,
						"Message" => $emailBody
					)
				);
		
				$sendEmailMessageResponse = $this->ApiRequest($request);
	
				$addConfirmationMessage('Richiesta di contatto inviata');
			}
		}

		return $output . $this->displayForm();
	}
	
	public function displayForm()
	{
		$fieldsValues = [];

		$formName = "";

		$getField = function($label, $fieldData = []) use (&$fieldsValues, &$formName) {
			$type = empty($fieldData["type"]) ? "text" : $fieldData["type"];
			$name = empty($fieldData["name"]) ? Tools::strtolower(str_replace(" ", "", $label)) : $fieldData["name"];

			$name = 'posteitaliane_' . $name;

			$fieldsValues[$formName][$name] = Configuration::get($name);

			$fieldArr = array(
				'type'     => $type,
				'label'    => $this->l($label),
				'name'     => $name,
				'required' => empty( $fieldData["required"] ) ? true : $fieldData["required"]
			);

			if ($type == 'select'){
				$fieldArr["multiple"] = $fieldData["multiple"];
				$fieldArr["options"] = [
					"query" => [],
					'id'   => 'id_option',
					'name' => 'name',
				];
				
				foreach ($fieldData["values"] as $selectIndex => $selectName)
				{
					$fieldIndex = $selectIndex + 1;

					$fieldArr["options"]["query"][] = array("id_option" => $fieldIndex, "name" => $selectName);
				}
			}

			if ($type == 'checkbox'){
				$fieldArr["values"] = [
					"query" => [],
					'id'   => 'id_option',
					'name' => 'name',
				];
				
				foreach ($fieldData["values"] as $checkboxIndex => $checkboxName)
				{
					$fieldIndex = $checkboxIndex + 1;
					$fieldValueName = $name . "_" . $fieldIndex;

					$fieldArr["values"]["query"][] = array("id_option" => $fieldIndex, "name" => $checkboxName, "val" => 1);
					$fieldsValues[$formName][$fieldValueName] = Configuration::get($fieldValueName);
				}
			}

			if ($name == 'posteitaliane_cap') $fieldArr["maxlength"] = 5;

			return $fieldArr;
		};

		$checkoutCarriersOptions = [];

		foreach ($this->getCheckoutCarriers() as $checkoutCarrier) {
			$checkoutCarriersOptions[] = "Abilita " . $checkoutCarrier["name"] . " al checkout";
		}

		// Checkout Carriers

		$formName = "checkout_carriers_form";

		$checkout_carriers_form = array(
			'form' => array(
				'legend' => array(
					'title' => $this->l('Configura metodi di spedizione'),
					'icon' => 'icon-cogs'
				),
				'input' => array(
					$getField("Metodi di spedizione", ["type" => "checkbox", "values" => $checkoutCarriersOptions, "required" => false])
				),
				'submit' => array(
					'title' => $this->l('Conferma'),
				)
			),
		);

		// Settings

		$formName = "settings_form";

		$settings_form = array(
			'form' => array(
				'legend' => array(
					'title' => $this->l('Hai già le credenziali API? Aggiungi la configurazione qui sotto'),
					'icon' => 'icon-cogs'
				),
				'input' => array(
					$getField("Codice Postazione"),
					$getField("Codice Cliente"),
					$getField("Login"),
					$getField("Password", ["type" => "password"]),
					$getField("Servizio", ["type" => "select", "values" => $this->GetActivableServices(), "multiple" => false]),
				),
				'submit' => array(
					'title' => $this->l('Collega Account'),
				)
			),
		);

		// Request Account

		$formName = "request_account_form";

		$request_account_form = array(
			'form' => array(
				'legend' => array(
					'title' => $this->l('Richiedi un contatto commerciale'),
					'icon' => 'icon-cogs'
				),
				'input' => array(
					$getField("Seleziona Servizio", ["name" => "attivaservizio", "type" => "checkbox", "values" => $this->GetActivableServices()]),
					$getField("Nome e Cognome", ["name" => "nomecognome"]),
					$getField("Ragione Sociale"),
					$getField("Indirizzo"),
					$getField("Citta"),
					$getField("Cap"),
					$getField("Nazione"),
					$getField("Partita IVA"),
					$getField("Telefono"),
					$getField("Email")
				),
				'submit' => array(
					'title' => $this->l('Richiedi un contatto commerciale'),
				)
			),
		);

		$lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));

		$settingsHelper = new HelperForm();
		$settingsHelper->show_toolbar = false;
		$settingsHelper->table =  $this->table;
		$settingsHelper->default_form_language = $lang->id;
		$settingsHelper->submit_action = 'submit' . $this->name . 'Settings';
		$settingsHelper->fields_value = $fieldsValues['settings_form'];

		$requestAccountHelper = clone $settingsHelper;
		$requestAccountHelper->submit_action = 'submit' . $this->name . 'RequestAccount';
		$requestAccountHelper->fields_value = $fieldsValues['request_account_form'];

		$checkoutCarriersHelper = clone $settingsHelper;
		$checkoutCarriersHelper->submit_action = 'submit' . $this->name . 'CheckoutCarriers';
		$checkoutCarriersHelper->fields_value = $fieldsValues['checkout_carriers_form'];

		$deleteCarrierUrl = $this->context->shop->getBaseURL(true) . "modules/posteitaliane/delete-carrier.php";

		$this->context->smarty->assign(array(
			'carriers'         => $this->GetCarriers(),
			'deleteCarrierUrl' => $deleteCarrierUrl
		));

		if ($this->isConfigured() == self::ERR_STORE_ADDRESS_NOT_CONFIGURED) return $this->displayError("Configura il tuo indirizzo mittente prima di utilizzare il modulo Poste Italiane dalla pagina Parametri Negozio > Contatto > Negozi > Dettagli di contatto");
		else if ($this->isConfigured() == self::ERR_MODULE_CONFLICT) return $this->displayError("Un conflitto è stato trovato. Si prega di disattivare/disinstallare il modulo ShippyPro");

		return
		$this->display(__FILE__, '/views/templates/admin/services.tpl') .
		$checkoutCarriersHelper->generateForm(array($checkout_carriers_form)) .
		$settingsHelper->generateForm(array($settings_form)) .
		$requestAccountHelper->generateForm(array($request_account_form));
	}

	public function GetPickupCarriers()
	{
		$carrierObjs = [];

		foreach ($this->GetCarriers() as $carrier) {
			$mainKey = $carrier->CarrierName . "|" . $carrier->Credentials->postazione;
			if (isset($carrierObjs[$mainKey])) continue;

			$carrierObjs[$mainKey] = $carrier;
		}

		return $carrierObjs;
	}

	public function GetCarriers()
	{
		$carrierObjs = [];

		$request = array(
			"Method" => "GetCarriers",
			"Params" => array(
				"Credentials" => [
					"SDA" => ["postazione"],
					"POSTEITALIANE" => ["postazione"]
				],
				"Active" => true
			)
		);

		$getCarriersApiResponse = $this->ApiRequest($request);

		if (isset($getCarriersApiResponse->Carriers->POSTEITALIANE))
		{
			$posteItalianeCarriers = $getCarriersApiResponse->Carriers->POSTEITALIANE;
			foreach ($posteItalianeCarriers as $posteItalianeCarrierIndex => $posteItalianeCarrier) $posteItalianeCarriers[$posteItalianeCarrierIndex]->CarrierName = "POSTEITALIANE";

			$carrierObjs = array_merge($carrierObjs, $posteItalianeCarriers);
		}

		if (isset($getCarriersApiResponse->Carriers->SDA))
		{
			$sdaCarriers = $getCarriersApiResponse->Carriers->SDA;
			foreach ($sdaCarriers as $sdaCarrierIndex => $posteItalianeCarrier) $sdaCarriers[$sdaCarrierIndex]->CarrierName = "SDA";

			$carrierObjs = array_merge($carrierObjs, $sdaCarriers);
		}

		return $carrierObjs;
	}

	public function GetCarrierById($carrierId)
	{
		$filteredCarriers = array_filter($this->GetCarriers(), function($q) use ($carrierId) { return $q->CarrierID == $carrierId; });
		$filteredCarriers = array_values($filteredCarriers);

		return empty( $filteredCarriers[0] ) ? FALSE : $filteredCarriers[0];
	}

	public function GetStoreAddress()
	{		
		$senderState = new State((int)Configuration::get('PS_SHOP_STATE_ID'));
		$senderCountry = new Country((int)Configuration::get('PS_SHOP_COUNTRY_ID'));

		return array(
			"name"     => Configuration::get('PS_SHOP_NAME'),
			"company"  => Configuration::get('PS_SHOP_NAME'),
			"address1" => Configuration::get('PS_SHOP_ADDR1'),
			"address2" => Configuration::get('PS_SHOP_ADDR2'),
			"postcode" => Configuration::get('PS_SHOP_CODE'),
			"city"     => Configuration::get('PS_SHOP_CITY'),
			"country"  => $senderCountry->iso_code,
			"state"    => $senderState->iso_code,
			"phone"    => Configuration::get('PS_SHOP_PHONE'),
			"email"    => Configuration::get('PS_SHOP_EMAIL')
		);
	}

	public function hookDisplayAdminOrderLeft($params)
	{
		return $this->displayAdminOrderMain( $params );
	}

	public function hookDisplayAdminOrderMain($params)
	{
		return $this->displayAdminOrderMain( $params );
	}

	public function displayAdminOrderMain($params)
	{
		$idOrder = (int)$params['id_order'];

		// Get Shipped Orders

		$shippedOrders = $this->getShippedOrders($idOrder);

		// Get Pickups

		$pickups = [];

		if (!empty($shippedOrders))
		{
			$request = array(
				"Method" => "GetPickups",
				"Params" => 
				array(
					"OrderID" => $shippedOrders[0]->ordine_id
				)
			);
	
			$getPickupsApiResponse = $this->ApiRequest($request);
			$pickups = isset($getPickupsApiResponse->Pickups) ? $getPickupsApiResponse->Pickups : [];
		}

		$order = new Order($idOrder);
		$address = new Address((int)$order->id_address_delivery);
		$addressInvoice = new Address((int)$order->id_address_invoice);
		$state = new State((int)$address->id_state);
		$country = new Country((int)$address->id_country);
		$customer = new Customer((int)$address->id_customer);

		$trackingUrl = $this->context->shop->getBaseURL(true) . "modules/posteitaliane/get-tracking.php";
		$shipUrl = $this->context->shop->getBaseURL(true) . "modules/posteitaliane/ship.php";
		$bookPickupUrl = $this->context->shop->getBaseURL(true) . "modules/posteitaliane/book-pickup.php";
		$markOrderAsShippedUrl = $this->context->shop->getBaseURL(true) . "modules/posteitaliane/mark-order-as-shipped.php";
		$openTicketUrl = $this->context->shop->getBaseURL(true) . "modules/posteitaliane/open-ticket.php";
		
		$isCodOrder = stripos($order->payment, 'contrassegno') !== FALSE;

		// Note fix
		if (!isset($order->note)) $order->note = "";
		
		$this->context->smarty->assign(array(
			'order'                 => $order,
			'address'               => $address,
			'addressInvoice'		=> $addressInvoice,
			'state'                 => $state,
			'country'               => $country,
			'customer'              => $customer,
			'shippedOrders'         => $shippedOrders,
			'shippedOrdersCount'    => count($shippedOrders),
			'pickups'               => $pickups,
			'pickupsCount'          => count($pickups),
			'pickupAddress'         => (object)$this->GetStoreAddress(),
			'trackingUrl'           => $trackingUrl,
			'shipUrl'               => $shipUrl,
			'bookPickupUrl'         => $bookPickupUrl,
			'openTicketUrl'			=> $openTicketUrl,
			'markOrderAsShippedUrl' => $markOrderAsShippedUrl,
			'isCodOrder'			=> $isCodOrder,
			'cashOnDelivery'		=> $isCodOrder ? number_format($order->total_paid, 2, '.', '') : '',
			'carriers' 				=> $this->GetCarriers(),
			'pickupCarriers' 		=> $this->GetPickupCarriers(),
			'countriesOptions'      => $this->getCountriesOptions(),
			'ajax_token'            => Tools::getToken(false),
			'isConfigured'			=> $this->isConfigured(),
			'lang'					=> $this->getLang(),
		));

		return $this->display(__FILE__, '/views/templates/admin/template.tpl');		
    }

	function getFirstNameLastName($name)
	{
		$chunks = explode(" ", $name, 2);
	
		$firstName = $chunks[0];
		$lastName = isset($chunks[1]) ? $chunks[1] : "";
	
		return [$firstName, $lastName];
	}
	
	public function WriteLog($log)
	{
		file_put_contents(_PS_MODULE_DIR_ . $this->name . "/log.txt", $log . PHP_EOL, FILE_APPEND);
	}
	
	public function ApiRequest($params, $apiKey = NULL)
	{
		if ($apiKey === NULL) $apiKey = $this->GetApiKey();

		$data = json_encode($params);
		
		$curl = curl_init();
		curl_setopt($curl, CURLOPT_POST, 1);
		curl_setopt($curl, CURLOPT_HTTPHEADER, array(
			'Content-Type: application/json',
			'Content-Length: ' . Tools::strlen($data))
		);                
		curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
		curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);    
		curl_setopt($curl, CURLOPT_USERPWD, $apiKey);
		curl_setopt($curl, CURLOPT_URL, "https://www.shippypro.com/api");
		curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
		$json = curl_exec($curl);          
		curl_close($curl);
		
		return json_decode($json);
	}
	
	public function SystemApiRequest($params)
	{
		$data = json_encode($params);
		
		$curl = curl_init();
		curl_setopt($curl, CURLOPT_POST, 1);
		curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
		curl_setopt($curl, CURLOPT_URL, "https://pi.shippypro.com/posteitaliane.php");
		curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
		$json = curl_exec($curl);          
		curl_close($curl);
		
		return json_decode($json);
	}

	public function GetApiKey()
	{		
		$email = "posteitaliane" . Configuration::get('PS_SHOP_EMAIL');
		$apiKey = Configuration::get('posteitaliane_api_key');

		if ($apiKey != NULL) return $apiKey;
		else
		{
			$carrierNamesToInstall = array_column(array_filter($this->getCheckoutCarriers(), function($q) { return $q["enabled_default"] == TRUE; }), "name");
			$this->getAndInstallCarriers($carrierNamesToInstall);

			$response = $this->SystemApiRequest([
				"Method" => "GetCustomers",
				"Params" => [
					"Email" => $email
				]
			]);

			if (isset($response->Customers[0])) $apiKey = $response->Customers[0]->APIKeys[0];
			else
			{
				// Create User
				
				$storeAddress = $this->GetStoreAddress();

				$response = $this->SystemApiRequest([
					"Method" => "RegisterNewUser",
					"Params" => [
						"customer_name"                      => $this->getFirstNameLastName($storeAddress["name"])[0],
						"customer_surname"                   => $this->getFirstNameLastName($storeAddress["name"])[1],
						"customer_email"                     => $email,
						"customer_company"                   => $storeAddress["company"],
						"customer_phone"                     => $storeAddress["phone"],
						"customer_street"                    => $storeAddress["address1"],
						"customer_housenr"                   => "",
						"customer_zip"                       => $storeAddress["postcode"],
						"customer_city"                      => $storeAddress["city"],
						"customer_country"                   => $storeAddress["country"],
						"customer_vatno"                     => "",
						"customer_state"                     => $storeAddress["state"],
						"customer_otherfiscalid"             => "",
						"customer_invoice_name"              => "",
						"customer_invoice_surname"           => "",
						"customer_invoice_email"             => "",
						"customer_invoice_company"           => "",
						"customer_invoice_phone"             => "",
						"customer_invoice_street"            => "",
						"customer_invoice_housenr"           => "",
						"customer_invoice_zip"               => "",
						"customer_invoice_city"              => "",
						"customer_invoice_country"           => "",
						"customer_invoice_vatno"             => "",
						"customer_invoice_state"             => "",
						"customer_invoice_otherfiscalid"     => "",
						"customer_avatar_url"                => "",
						"customer_whitelabel_clientref"      => "",
						"customer_whitelabel_paywithcredits" => 0,
						"customer_show_net_rates"            => 0,
						"wholesaler"                         => false,
						"create_apikey"						 => true
					]
				]);

				$apiKey = $response->APIKeys[0];

				// Register website

				$response = $this->SystemApiRequest([
					"Method" => "NewMap",
					"Params" => [
						"domain"      => str_replace(array('http://','https://'), '', _PS_BASE_URL_),
						"marketplace" => "Prestashop"
					]
				]);
			}

			Configuration::updateValue('posteitaliane_api_key', $apiKey);
		}

		return $apiKey;
	}

	public function GetActivableServices()
	{
		return ['POSTEITALIANE - Crono', 'POSTEITALIANE - Crono Economy', 'POSTEITALIANE - Crono Internazionale', 'POSTEITALIANE - Crono Reverse', 'POSTEITALIANE - Crono Express', 'SDA - Extralarge'];
	}

	public function ThrowAPIResponse($response)
	{
		header('Content-Type: application/json');
		echo json_encode($response);
		exit(200);
	}
	
	public function hookDisplayBeforeCarrier($params)
	{
		$setCartPointUrl = $this->context->shop->getBaseURL(true) . "modules/posteitaliane/save-cart-point.php";

		$id_address_delivery = (int)$params["cart"]->id_address_delivery;
		$address = new Address($id_address_delivery);
		$country = new Country((int)$address->id_country);

		$this->context->smarty->assign(array(
			'setCartPointUrl'    => $setCartPointUrl,
			'country'			 => $country->iso_code,
			'city'				 => $address->city,
			'postcode'			 => $address->postcode,
			'lang'			 	 => $this->GetLang()
        ));

		return $this->display(__FILE__, '/views/templates/front/checkout.tpl');
	}

	public function hookActionValidateOrder($params)
    {
        if ($this->context->cookie->useAccessPoint)
        {
			$address = new Address($params["order"]->id_address_delivery);
            $address->id = 0;
            $address->company = $this->context->cookie->accessPointName;
            $address->address1 = $this->context->cookie->accessPointAddress1;
            $address->city = $this->context->cookie->accessPointCity;
            $address->postcode = $this->context->cookie->accessPointPostcode;
            $address->alias = Tools::substr("Access Point - " . $this->context->cookie->accessPointName, 0, 32);
            $address->save();

            $params["order"]->id_address_delivery = $address->id;
			$params["order"]->update();
			
			$address->delete();

			$urlCart = str_replace('www.', '', $this->context->cookie->urlCart);
			
            $request = array(
                'Method' => 'ChoosePoint',
                'Params' => array(
                    'url'         => $urlCart,
                    'selected_id' => $this->context->cookie->pointSelectedId,
                    'order_id'    => $params["order"]->id
                )
			);

			$choosePointApiRequest = $this->SystemApiRequest($request);

            $this->context->cookie->useAccessPoint = false;
        }
	}
	
	public function isConfigured()
	{
		$storeAddress = $this->GetStoreAddress();

		if (
			empty($storeAddress["name"]) || 
			empty($storeAddress["company"]) || 
			empty($storeAddress["address1"]) || 
			empty($storeAddress["city"]) || 
			empty($storeAddress["postcode"]) || 
			empty($storeAddress["country"]) || 
			empty($storeAddress["phone"]) || 
			empty($storeAddress["email"])
		) return self::ERR_STORE_ADDRESS_NOT_CONFIGURED;
		else if (
			Module::isInstalled('shippypro') && Module::isEnabled('shippypro')
		) return self::ERR_MODULE_CONFLICT;
		else return self::CONFIGURED;
	}

	public function getLang()
	{
		$lang = Tools::strtoupper(Tools::substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2));

		if (!in_array($lang, ["IT", "EN"])) $lang = "EN";

		return $lang;
	}
	
	public function uninstallCarriers()
	{		
		$carriers = Carrier::getCarriers($this->context->cookie->id_lang, true, false, false, NULL, PS_CARRIERS_AND_CARRIER_MODULES_NEED_RANGE);
		
		foreach (array_filter($carriers) as $carrier)
		{
			if ($carrier["external_module_name"] == $this->name)
			{
				$carrierObj = new Carrier($carrier['id_carrier']);
				$carrierObj->deleted = 1;
				$carrierObj->update();
			}
		}
	}
	
	public function uninstallCarrier($carrierName)
	{		
		$carriers = Carrier::getCarriers($this->context->cookie->id_lang, true, false, false, NULL, PS_CARRIERS_AND_CARRIER_MODULES_NEED_RANGE);
		
		foreach (array_filter($carriers) as $carrier)
		{
			if ($carrier["external_module_name"] == $this->name && $carrier["name"] == $carrierName)
			{
				$carrierObj = new Carrier($carrier['id_carrier']);
				$carrierObj->deleted = 1;
				$carrierObj->update();
			}
		}
	}

	public function getCheckoutCarriers() {
		$checkoutCarriers = [
			["name" => "Poste Italiane - Consegna a Domicilio", "delay" => "Spedizione a casa", "enabled_default" => true],
			["name" => "Poste Italiane - Punto di Ritiro", "delay" => "Ritiro in negozio", "enabled_default" => false],
		];

		foreach ($checkoutCarriers as $checkoutCarrierIndex => $checkoutCarrier) {
			$checkoutCarriers[$checkoutCarrierIndex]["configuration_key_name"] = "posteitaliane_metodidispedizione_" . ($checkoutCarrierIndex + 1);
		}

		return $checkoutCarriers;
	}
	
	public function getAndInstallCarriers($carrierNamesToInstall = [])
	{
		$carriersToInstall = array();

		foreach ($this->getCheckoutCarriers() as $checkoutCarrier) {
			if (!in_array($checkoutCarrier["name"], $carrierNamesToInstall)) continue;

			$psCarrierName                          = $checkoutCarrier["name"];
			$carrierToInstall                       = new stdClass();
			$carrierToInstall->Name                 = $psCarrierName;
			$carrierToInstall->CarrierPSName        = $psCarrierName;
			$carrierToInstall->DeliveryDays         = $checkoutCarrier["delay"];
			$carrierToInstall->ConfigurationKeyName = $checkoutCarrier["configuration_key_name"];
			$carriersToInstall[]					= $carrierToInstall;
		}

		$installedCarriers = $this->getInstalledCarriers();

		foreach ($carriersToInstall as $carrier)
		{			
			if (in_array($carrier->Name, $installedCarriers)) continue;
			
			$carrierConfig = array(
				0 => array(
					'name'                 => $carrier->CarrierPSName,
					'id_tax_rules_group'   => 0,
					'active'               => true,
					'deleted'              => 0,
					'shipping_handling'    => false,
					'range_behavior'       => 0,
					'delay'                => array('fr' => $carrierToInstall->DeliveryDays, 'en' => $carrierToInstall->DeliveryDays, Language::getIsoById(Configuration::get('PS_LANG_DEFAULT')) => $carrierToInstall->DeliveryDays),
					'id_zone'              => 1,
					'is_module'            => true,
					'shipping_external'    => true,
					'external_module_name' => $this->name,
					'need_range'           => true
				)
			);

			$id_carrier = $this->installExternalCarrier($carrierConfig[0], "POSTEITALIANE");
			Configuration::updateValue($carrier->ConfigurationKeyName, 1);
		}
	}
	
	public function installExternalCarrier($config, $carrierName)
	{
		$carrier = new Carrier();
		$carrier->name = $config['name'];
		$carrier->id_tax_rules_group = $config['id_tax_rules_group'];
		$carrier->id_zone = $config['id_zone'];
		$carrier->active = $config['active'];
		$carrier->deleted = $config['deleted'];
		$carrier->delay = $config['delay'];
		$carrier->shipping_handling = $config['shipping_handling'];
		$carrier->range_behavior = $config['range_behavior'];
		$carrier->is_module = $config['is_module'];
		$carrier->shipping_external = $config['shipping_external'];
		$carrier->external_module_name = $config['external_module_name'];
		$carrier->need_range = $config['need_range'];
		$carrier->url = "https://www.poste.it/cerca/index.html#/risultati-spedizioni/@";
		$languages = Language::getLanguages(true);

		foreach ($languages as $language)
		{
			if ($language['iso_code'] == 'fr')
				$carrier->delay[(int)$language['id_lang']] = $config['delay'][$language['iso_code']];
			if ($language['iso_code'] == 'en')
				$carrier->delay[(int)$language['id_lang']] = $config['delay'][$language['iso_code']];
			if ($language['iso_code'] == Language::getIsoById(Configuration::get('PS_LANG_DEFAULT')))
				$carrier->delay[(int)$language['id_lang']] = $config['delay'][$language['iso_code']];
		}

		if ($carrier->add())
		{
			$groups = Group::getGroups(true);
			foreach ($groups as $in => $group)
			{
				Db::getInstance()->insert('carrier_group', array(
					'id_carrier' => (int)$carrier->id,
					'id_group'   => (int)$group['id_group']
				));
			}
			
			$rangePrice = new RangePrice();
			$rangePrice->id_carrier = $carrier->id;
			$rangePrice->delimiter1 = '0';
			$rangePrice->delimiter2 = '10000';
			$rangePrice->add();
			$rangeWeight = new RangeWeight();
			$rangeWeight->id_carrier = $carrier->id;
			$rangeWeight->delimiter1 = '0';
			$rangeWeight->delimiter2 = '10000';
			$rangeWeight->add();
			$zones = Zone::getZones(true);
			foreach ($zones as $zone)
			{
				Db::getInstance()->insert('carrier_zone', array(
					'id_carrier' => (int)$carrier->id,
					'id_zone'    => (int)$zone['id_zone']
				));
				
				Db::getInstance()->insert('delivery', array(
					'id_carrier'      => (int)$carrier->id,
					'id_range_price'  => (int)$rangePrice->id,
					'id_range_weight' => null,
					'id_zone'         => (int)$zone['id_zone'],
					'price'           => 0
				));
				
				Db::getInstance()->insert('delivery', array(
					'id_carrier'      => (int)$carrier->id,
					'id_range_price'  => null,
					'id_range_weight' => (int)$rangePrice->id,
					'id_zone'         => (int)$zone['id_zone'],
					'price'           => 0
				));
			}
			
			$carrierLogo    = _PS_SHIP_IMG_DIR_ . '/' . (int)$carrier->id . '.png';
			$carrierLogoJPG = _PS_SHIP_IMG_DIR_ . '/' . (int)$carrier->id . '.jpg';
			$imageCarrierName = $carrierName;

			// Copy Logo
			if (!copy("https://cdn.shippypro.com/assets/images/carriers/" . Tools::strtolower($imageCarrierName) . ".png", $carrierLogo))
				return false;
			
			$this->resizeImage($carrierLogo, $carrierLogoJPG, 100, 40, 40, "png");
			
			unlink($carrierLogo);

			// Return ID Carrier
			return (int)($carrier->id);
		}
	}
	
	public function resizeImage($originalImage, $outputImage, $quality, $maxWidth, $maxHeight, $ext)
	{
		if (preg_match('/jpg|jpeg/i',$ext))
			$imageTmp=imagecreatefromjpeg($originalImage);
		else if (preg_match('/png/i',$ext))
			$imageTmp=imagecreatefrompng($originalImage);
		else if (preg_match('/gif/i',$ext))
			$imageTmp=imagecreatefromgif($originalImage);
		else if (preg_match('/bmp/i',$ext))
			$imageTmp=imagecreatefrombmp($originalImage);
		else
			return 0;
		// Get dimensions of source image.
		list($origWidth, $origHeight) = getimagesize($originalImage);
		if ($maxWidth == 0)
		{
			$maxWidth  = $origWidth;
		}
		if ($maxHeight == 0)
		{
			$maxHeight = $origHeight;
		}
		// Calculate ratio of desired maximum sizes and original sizes.
		$widthRatio = $maxWidth / $origWidth;
		$heightRatio = $maxHeight / $origHeight;
		// Ratio used for calculating new image dimensions.
		$ratio = min($widthRatio, $heightRatio);
		// Calculate new image dimensions.
		$newWidth  = (int)$origWidth  * $ratio;
		$newHeight = (int)$origHeight * $ratio;
		// Create final image with new dimensions.
		$newImage = imagecreatetruecolor($newWidth, $newHeight);
		$whiteBackground = imagecolorallocate($newImage, 255, 255, 255); 
		imagefill($newImage,0,0,$whiteBackground); // fill the background with white
		imagecopyresampled($newImage, $imageTmp, 0, 0, 0, 0, $newWidth, $newHeight, $origWidth, $origHeight);
		imagejpeg($newImage, $outputImage, $quality);
		// Free up the memory.
		imagedestroy($imageTmp);
		imagedestroy($newImage);
		return 1;
	}

	public function getInstalledCarriers() {
		$carrierObjs = [];

		$carriers = Carrier::getCarriers($this->context->cookie->id_lang, true, false, false, NULL, PS_CARRIERS_AND_CARRIER_MODULES_NEED_RANGE);
		
		foreach (array_filter($carriers) as $carrier)
		{
			if ($carrier["external_module_name"] == $this->name) $carrierObjs[] = $carrier["name"];
		}

		return $carrierObjs;
	}

	public function getOrderShippingCost($params, $original_shipping_cost)
	{
		return $original_shipping_cost;
	}
	
	public function getOrderShippingCostExternal($params) { }

	public function getCountriesOptions() {
		$countries = [
			"CA" => "Canada",
			"US" => "Stati Uniti",
			"GB" => "Regno Unito",
			"AF" => "Afghanistan",
			"AL" => "Albania",
			"DZ" => "Algeria",
			"AD" => "Andorra",
			"AO" => "Angola",
			"AR" => "Argentina",
			"AM" => "Armenia",
			"AW" => "Aruba",
			"AG" => "Antigua And Barbuda",
			"AU" => "Australia",
			"AT" => "Austria",
			"AZ" => "Azerbaijan",
			"BD" => "Bangladesh",
			"BS" => "Bahamas",
			"BH" => "Bahrain",
			"BB" => "Barbados",
			"BY" => "Belarus",
			"BE" => "Belgio",
			"BZ" => "Belize",
			"BJ" => "Benin",
			"BM" => "Bermuda",
			"BT" => "Bhutan",
			"BO" => "Bolivia",
			"BA" => "Bosnia And Herzegovina",
			"BW" => "Botswana",
			"BR" => "Brasile",
			"BN" => "Brunei",
			"BG" => "Bulgaria",
			"NC" => "Nuova Caledonia",
			"KH" => "Cambogia",
			"CM" => "Republic of Cameroon",
			"KY" => "Isole Cayman",
			"CL" => "Cile",
			"CN" => "Cina",
			"CO" => "Colombia",
			"KM" => "Comoros",
			"CG" => "Congo",
			"CD" => "Congo, Democratic Republic",
			"CR" => "Costa Rica",
			"CI" => "Côte d'Ivoire",
			"HR" => "Croazia",
			"CW" => "Curaçao",
			"CY" => "Cipro",
			"CZ" => "Repubblica Ceca",
			"DK" => "Danimarca",
			"DM" => "Dominica",
			"DO" => "Repubblica Dominicana",
			"EC" => "Ecuador",
			"EG" => "Egitto",
			"SV" => "El Salvador",
			"EE" => "Estonia",
			"ET" => "Etiopia",
			"FO" => "Isole Faroe",
			"FJ" => "Fiji",
			"FI" => "Finlandia",
			"FR" => "Francia",
			"PF" => "Polinesia francese",
			"GM" => "Gambia",
			"GE" => "Georgia",
			"DE" => "Germania",
			"GH" => "Ghana",
			"GI" => "Gibilterra",
			"GR" => "Grecia",
			"GL" => "Greenland",
			"GD" => "Grenada",
			"GP" => "Guadeloupe",
			"GT" => "Guatemala",
			"GG" => "Guernsey",
			"GY" => "Guyana",
			"HT" => "Haiti",
			"HN" => "Honduras",
			"HK" => "Hong Kong",
			"HU" => "Ungheria",
			"IS" => "Islanda",
			"IN" => "India",
			"ID" => "Indonesia",
			"IE" => "Irlanda",
			"IM" => "Isle Of Man",
			"IL" => "Israele",
			"IT" => "Italia",
			"JM" => "Jamaica",
			"JP" => "Giappone",
			"JE" => "Jersey",
			"JO" => "Giordania",
			"KZ" => "Kazakhstan",
			"KE" => "Kenya",
			"XK" => "Kosovo",
			"KW" => "Kuwait",
			"KG" => "Kyrgyzstan",
			"LV" => "Lettonia",
			"LB" => "Libano",
			"LS" => "Lesotho",
			"LR" => "Liberia",
			"LI" => "Liechtenstein",
			"LT" => "Lituania",
			"LU" => "Lussemburgo",
			"MO" => "Macao",
			"MK" => "Macedonia, Republic Of",
			"MG" => "Madagascar",
			"MW" => "Malawi",
			"MY" => "Malesia",
			"MV" => "Maldive",
			"MT" => "Malta",
			"MQ" => "Martinica",
			"MU" => "Mauritius",
			"MX" => "Messico",
			"MD" => "Moldova, Republic of",
			"MC" => "Monaco",
			"MN" => "Mongolia",
			"ME" => "Montenegro",
			"MA" => "Marocco",
			"MZ" => "Mozambico",
			"MM" => "Myanmar",
			"NA" => "Namibia",
			"NP" => "Nepal",
			"AN" => "Antille olandesi",
			"NL" => "Olanda",
			"NZ" => "Nuova Zelanda",
			"NI" => "Nicaragua",
			"NE" => "Niger",
			"NG" => "Nigeria",
			"NO" => "Norvegia",
			"OM" => "Oman",
			"PK" => "Pakistan",
			"PS" => "Palestinian Territory, Occupied",
			"PA" => "Panama",
			"PG" => "Papua Nuova Guinea",
			"PY" => "Paraguay",
			"PE" => "Perù",
			"PH" => "Filippine",
			"PL" => "Polonia",
			"PT" => "Portogallo",
			"QA" => "Qatar",
			"RE" => "Riunione",
			"RO" => "Romania",
			"RU" => "Russia",
			"RW" => "Ruanda",
			"KN" => "Saint Kitts And Nevis",
			"LC" => "Santa Lucia",
			"SX" => "Saint Martin",
			"ST" => "Sao Tome And Principe",
			"WS" => "Samoa",
			"SA" => "Arabia saudita",
			"SN" => "Senegal",
			"RS" => "Serbia",
			"SC" => "Seychelles",
			"SG" => "Singapore",
			"SK" => "Slovacchia",
			"SI" => "Slovenia",
			"ZA" => "Sud Africa",
			"KR" => "Korea del Sud",
			"ES" => "Spagna",
			"LK" => "Sri Lanka",
			"VC" => "St. Vincent",
			"SR" => "Suriname",
			"SE" => "Svezia",
			"CH" => "Svizzera",
			"SY" => "Siria",
			"TW" => "Taiwan",
			"TH" => "Tailandia",
			"TZ" => "Tanzania, United Republic Of",
			"TT" => "Trinidad e Tobago",
			"TN" => "Tunisia",
			"TR" => "Turchia",
			"TM" => "Turkmenistan",
			"TC" => "Isole Turks e Caicos",
			"UG" => "Uganda",
			"UA" => "Ucraina",
			"AE" => "Emirati Arabi Uniti",
			"UY" => "Uruguay",
			"UZ" => "Uzbekistan",
			"VU" => "Vanuatu",
			"VE" => "Venezuela",
			"VN" => "Vietnam",
			"VI" => "Virgin Islands, British",
			"YE" => "Yemen",
			"ZM" => "Zambia",
			"ZW" => "Zimbabwe",
			"AX" => "Isole Aland",
			"AI" => "Anguilla",
			"BV" => "Isole Bouvet",
			"IO" => "Territorio bitannico dell'oceano Indiano",
			"BF" => "Burkina Faso",
			"BI" => "Burundi",
			"CV" => "Cape Verde",
			"CF" => "Repubblica Centrafricana",
			"TD" => "Chad",
			"CX" => "Isola di Natale",
			"CC" => "Isole Cocos (Keeling)",
			"CK" => "Isole Cook",
			"CU" => "Cuba",
			"DJ" => "Gibuti",
			"GQ" => "Guinea Equatoriale",
			"ER" => "Eritrea",
			"FK" => "Falkland Islands (Malvinas)",
			"GF" => "Guiana francese",
			"TF" => "Territori francesi del sud",
			"GA" => "Gabon",
			"GN" => "Guinea",
			"GW" => "Guinea Bissau",
			"HM" => "Heard Island And Mcdonald Islands",
			"VA" => "Holy See (Vatican City State)",
			"IR" => "Iran, Islamic Republic Of",
			"IQ" => "Iraq",
			"KI" => "Kiribati",
			"KP" => "Korea, Democratic People's Republic Of",
			"LA" => "Lao People's Democratic Republic",
			"LY" => "Libyan Arab Jamahiriya",
			"ML" => "Mali",
			"MR" => "Mauritania",
			"YT" => "Mayotte",
			"MS" => "Montserrat",
			"NR" => "Nauru",
			"NU" => "Niue",
			"NF" => "Isola Norfolk",
			"PN" => "Isole Pitcairn",
			"BL" => "Saint-Barthélemy",
			"SH" => "Sant'Elena",
			"PM" => "Saint Pierre And Miquelon",
			"SM" => "San Marino",
			"SL" => "Sierra Leone",
			"SB" => "Isole Salomone",
			"SO" => "Somalia",
			"GS" => "South Georgia And The South Sandwich Islands",
			"SD" => "Sudan",
			"SJ" => "Svalbard And Jan Mayen",
			"SZ" => "Swaziland",
			"TJ" => "Tagikistan",
			"TL" => "Timor Leste",
			"TG" => "Togo",
			"TK" => "Tokelau",
			"TO" => "Tonga",
			"TV" => "Tuvalu",
			"UM" => "Isole Minori Esterne degli Stati Uniti",
			"WF" => "Wallis And Futuna",
			"EH" => "Sahara ovest",
			"PR" => "Porto Rico",
			"IC" => "Canary Island",
			"GU" => "Guam"
		];

		return $countries;
	}

	public function getShippedOrders($idOrder) {
		$request = array(
			"Method" => "GetShippedOrders",
			"Params" => 
			array(
				"CarrierName" => ["POSTEITALIANE", "SDA"],
				"TransactionID" => "ORDER" . $idOrder,
				"OrderStatus" => 1
			)
		);
		
		$shippedOrdersApiResponse = $this->ApiRequest($request);
		$shippedOrders = isset($shippedOrdersApiResponse->Orders) ? $shippedOrdersApiResponse->Orders : [];

		return $shippedOrders;
	}
}
